/* 
 * Greeter IX-Agent
 * Jussi Stader - adapted from Thomas French, 05/03/07
 */

package ix.secondlife;

import ix.util.*;
import ix.util.PatternParser.LispPatternParser;
import ix.util.lisp.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.iface.util.*;
import ix.iface.domain.*;
import ix.iview.util.IVUtil;
import ix.ichat.*;

import java.util.*;
import java.util.concurrent.ConcurrentLinkedQueue;

public class GreeterAgent extends IXAgent {
    
    private final String m_destPP = "process-panel";
    private final String m_myName = "greeter-agent";

    private Thread msgCoord = null;

    private ConcurrentLinkedQueue<String> m_queue;
    private Boolean m_waitingResponse = Boolean.valueOf("false");
    private Boolean m_lastMsgPoll = Boolean.valueOf("false");

    private GreeterAgent m_myself;

    private Name m_activityRef;

    // Unique SL id.
    private final String CHANNEL_ID_PARAM = "sl-channel-id";
    private final String XMLRPC_SERVER = "sl-xmlrpc-server";
    private String m_object_uid;
    private String m_xmlrpc_server;

    // Constructor
    public GreeterAgent() {
        super();
        
        m_myself = this;
        
        m_queue = new ConcurrentLinkedQueue<String>();
    }
    
     // Next few methods copied from JD's IQuery code:

     /**
      * Command-line argument processing.
      */
     protected void processCommandLineArguments() {
	 super.processCommandLineArguments();
     }

     /**
      * Agent setup and initialization.
      * The method supplied by the IQuery class just
      * calls {@link #setupGUI()} in the AWT event thread.
      *
      * @see IXAgent#startup()
      */
     protected void startup() {
	 Util.swingAndWait(new Runnable() {
	     public void run() {
	       setupGUI();
	     }
	   });
     }

     /**
      * Creates a "transcript" frame for this agent, including
      * an "Exit" button.
      */
     protected void setupGUI() {
	 // This is optional.  We could just let the first incoming message
	 // cause the text frame to be created.  But this way we can make it
	 // appear right away and can add an "Exit" button.

	 String framename = ipcName;

         // Get properties
         if(Parameters.haveParameter(CHANNEL_ID_PARAM))
         {
             m_object_uid = Parameters.getParameter(CHANNEL_ID_PARAM);
             System.out.println("Second Life Channel ID: " + m_object_uid);
         }
         
         if(Parameters.haveParameter(XMLRPC_SERVER) )
         {
             m_xmlrpc_server = Parameters.getParameter(XMLRPC_SERVER);
             System.out.println("Second Life XML-RPC Server: " + m_xmlrpc_server);
         }

	 textFrame = new TextAreaFrame("Messages for " + framename,
				       new String[] { "Exit" } );
         //new String[] { "Start SL Comms", "Stop SL Comms", "Exit" } );

	 textFrame.setFoldLongLines(0);
	 textFrame.addToolManager();
	 textFrame.addListener(new TextAreaFrame.TListener() {
	     public void buttonPressed(String action) {
		 if (action.equals("Exit"))
		 {
		     if (Util.dialogConfirms
			     (textFrame, "Are you sure you want to exit?"))
                     {
                         if ( msgCoord != null && msgCoord.isAlive() )
                         {
                             textFrame.append("Stopping communication monitor ... ");
                             try
                             {
                                 // interrupt thread
                                 msgCoord.interrupt();
                                 
                                 //wait indefinitely for thread to die
                                 msgCoord.join();
                                 
                                 msgCoord = null;
                                 
                                 textFrame.append("Done.\n");
                             }                     
                             catch (InterruptedException ie)
                             {
                                 ie.printStackTrace();
                             }
                         }
                         
                         exit();
                     }
                }
             }});
                 /*
		 else if (action.equals("Start SL Comms"))
                {
                  textFrame.append("Starting SL communication monitor ... ");
                                   
                  try 
                  {
		      //Start new SL comms thread monitoring
		      msgCoord = new MessageCoordinator(m_myself);
		      msgCoord.start();

		      textFrame.append("Done.\n");
                  }
                  catch( Exception x ) 
                  {
                    x.printStackTrace();
                  }	 
                }
                else if ( action.equals("Stop SL Comms") )
                {
                  try
                  {
                    if ( msgCoord != null && msgCoord.isAlive() )
                    {
                      textFrame.append("Stopping communication monitor ... ");
                      // interrupt thread
		      msgCoord.interrupt();

                      //wait indefinitely for thread to die
		      msgCoord.join();
                      
                      msgCoord = null;

                      textFrame.append("Done.\n");
                    }
                  }
                  catch (InterruptedException ie)
                  {
                    ie.printStackTrace();
                  }
                }
                
                }});
                 */   
                               
         // Startup thread to handle communication to SL.
         try 
         {
             //Start new SL comms thread monitoring
             msgCoord = new MessageCoordinator(m_myself);
             msgCoord.start();
             
             //textFrame.append("Communication monitor to SL started.\n");
         }
         catch( Exception x ) 
         {
             x.printStackTrace();
         }	 
         
         outputText(m_myName + " ready...\n");
    }

    /**
     * Called when the agent should cease execution.
     */
    public void exit() {
	System.exit(0);
    }

    public void addTool(ToolController tc) {
	textFrame.addTool(tc);
    }


    public static void main(String[] args){
	new GreeterAgent().mainStartup(args);
    }

    public void handleNewActivity(Activity activity){
	//Debug.noteln("Received activity");

	m_activityRef = activity.getRef();

	LList lst = activity.getPattern();

	LispPatternParser lpp = new LispPatternParser();

	String act = lpp.unparsePattern(lst);

	//System.out.println("Received activity: " + act);

	// Add activity to queue for sending to SL.
	m_queue.add(act);
    }

    public void handleNewIssue(Issue issue){
	Debug.noteln("Received issue");
    }

    public void handleNewConstraint(Constraint constraint){
	Debug.noteln("Received constraint");
    }

    public void handleNewReport(Report report){
	Debug.noteln("Received report");
    }

    public void handleNewChatMessage(ChatMessage message){
	Debug.noteln("Received chat message");
    }

    public void sendNewActivity(String activityText)
    {
        textFrame.append(activityText + "\n");
        
        IPC.getCommunicationStrategy().sendObject(m_destPP, 
                                                  new Activity(activityText)); 
    }

    public void sendNewReport(String reportText)
    {
	textFrame.append(reportText + "\n");

	// note: this says that all reports are of events...
	IPC.getCommunicationStrategy().sendObject(m_destPP, 
				       new Report(ReportType.EVENT,reportText)); 
    }

    public void sendNewIssue(String issueText)
    {
        textFrame.append(issueText + "\n");
        
        IPC.getCommunicationStrategy().sendObject(m_destPP, 
                                                  new Issue(issueText)); 
    }

    public void sendNewConstraint(String first, String second, String third)
    {
        textFrame.append("Send Constraint: ("+first+", "+second+", "
                         +third+")\n");
        
        LListCollector pattern = new LListCollector();
        pattern.add(Symbol.intern(first));
        pattern.add(Symbol.intern(second));
        PatternAssignment pa = new PatternAssignment(pattern.contents(),
                                                     Symbol.intern(third));
        LListCollector lonp = new LListCollector();
        lonp.add(pa);
        
        IPC.getCommunicationStrategy().sendObject(m_destPP, 
                                                  new Constraint("world-state", 
                                                                 "effect", 
                                                                 lonp.contents()));
    }

    // Called by Message Coordinator to get next message to send.
    public String getNextMessage()
    { 
        // don't remove item (just peek), 
        // only remove when are sure we can send it.
        String nextMsg = m_queue.peek();

        synchronized(m_waitingResponse)
        {
            synchronized(m_lastMsgPoll)
            {
                // don't overwrite messages, wait for reply.
                if ( !m_waitingResponse.booleanValue() )
                {
                    if ( nextMsg == null ) return "poll";
                    
                    m_queue.poll();
                    return nextMsg;
                }
                // messages from I-X are more important than poll msgs.
                else if ( m_waitingResponse.booleanValue() 
                     && m_lastMsgPoll.booleanValue() )
                {
                    if ( nextMsg == null || nextMsg.equals("poll") ) return null; 

                    m_queue.poll();
                    return nextMsg;
                }
            }
        }
        return null;
    }

    // Should we send poll message?
    // Only if no messages are waiting to be sent,
    // and not waiting for a reply from a previous message.
    private boolean sendPoll()
    {
        if ( m_queue.peek() != null || getWaitingResponse() )
        {            
            return false;
        }            
        return true;
    }

    // Set waiting for a response. Used to prevent multiple messages 
    // being sent to SL. SL only queues one message.
    public void setWaitingResponse(boolean b)
    {
        synchronized(m_waitingResponse)
        {
            m_waitingResponse = Boolean.valueOf(b);
        }
    }

    // Get value of waiting for a response.
    public boolean getWaitingResponse()
    {
        synchronized(m_waitingResponse)
        {
            return m_waitingResponse.booleanValue();
        }
    }

    // Send progress report to IX Process Panel.
    public void sendProgressReport (String reportText)
    {
	textFrame.append("EVENT: " + reportText + "\n");

	Report rp = new Report(ReportType.EVENT, reportText);
	//rp.setRef(m_activityRef);

	IPC.getCommunicationStrategy().sendObject(m_destPP, rp);

        // Keep poll messages stacked up so SL can send back message.
	sendPollMessage();
    }

    // Send constraint to IX Process Panel.
    public void sendConstraint (String reportText)
    {
      textFrame.append("CONSTRAINT: " + reportText + "\n");

      try {
	Constraint constraint = 
	    IVUtil.readCondition(Symbol.valueOf("world-state"), 
				 Symbol.valueOf("effect"),
				 reportText);
	IPC.getCommunicationStrategy().sendObject(m_destPP, constraint);
      }
      catch (SyntaxException e) {
	  textFrame.append(" ERROR: illegal constraint  - ignoring it");
	  Debug.describeException(e);
      }

      // Keep poll messages stacked up so SL can send back message.
      sendPollMessage();
    }

  // Send completion report to IX Process Panel.
    public void sendCompletionReport (String reportText)
    {
	textFrame.append("SUCCESS: " + reportText + "\n");

	Report rp = new Report(ReportType.SUCCESS, reportText);
	rp.setRef(m_activityRef);

	IPC.getCommunicationStrategy().sendObject(m_destPP, rp);

        // Keep poll messages stacked up so SL can send back message.
	sendPollMessage();
    }

    // Hack. In order to maintain possibility for greeter to reply.
    // Would be better to use HTTP requests from SL out.    
    public void sendPollMessage()
    {
	if ( sendPoll() )
        {
            m_queue.add("poll");
        }
    }

    // Get xml-rpc server address.
    public String getXMLRPCServer()
    {
        return m_xmlrpc_server;
    }

    // Get SL UID for greeter object.
    public String getObjectUID()
    {
        return m_object_uid;
    }

    public void outputText(String txt)
    {
        textFrame.append(txt + "\n");
    }
}
